<?php
// Arquivo: connections_dashboard.php
$dbFile = __DIR__ . '/database.db';

// Função para conectar ao banco de dados SQLite
function getDBConnection($dbFile) {
    try {
        return new PDO("sqlite:$dbFile");
    } catch (PDOException $e) {
        die("Erro ao conectar ao banco de dados: " . $e->getMessage());
    }
}

// Processar ação de limpar dados
if (isset($_POST['clear_data'])) {
    try {
        $db = getDBConnection($dbFile);
        $db->exec("DELETE FROM user_connections");
        $success_message = "Dados limpos com sucesso!";
    } catch (PDOException $e) {
        $error = "Erro ao limpar dados: " . $e->getMessage();
    }
}

// Obtém todas as conexões
$connections = [];
try {
    $db = getDBConnection($dbFile);
    $stmt = $db->query("SELECT * FROM user_connections ORDER BY last_connection DESC");
    $connections = $stmt->fetchAll(PDO::FETCH_ASSOC);
} catch (PDOException $e) {
    $error = "Erro ao buscar conexões: " . $e->getMessage();
}
?>
<?php include './includes/header.php'; ?>

<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Dashboard de Conexões</title>
    <link href="https://cdn.jsdelivr.net/npm/font-awesome/css/font-awesome.min.css" rel="stylesheet">
    <style>
        :root {
            --primary-color: #3498db;
            --online-color: #2ecc71;
            --offline-color: #e74c3c;
            --card-bg: #ffffff;
            --text-color: #34495e;
            --light-text: #7f8c8d;
            --border-color: #ecf0f1;
            --header-height: 60px;
            --mobile-padding: 15px;
            --danger-color: #e74c3c;
        }
        
        * {
            box-sizing: border-box;
        }
        
        body {
            font-family: 'Segoe UI', Tahoma, Geneva, Verdana, sans-serif;
            background-color: #f8f9fa;
            margin: 0;
            padding: 0;
            color: var(--text-color);
            line-height: 1.6;
        }
        
        .container {
            width: 100%;
            max-width: 1200px;
            margin: 0 auto;
            padding: 20px;
        }
        
        h1 {
            color: var(--text-color);
            text-align: center;
            margin: 20px 0 30px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 15px;
            font-size: 1.8rem;
        }
        
        .stats-container {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(250px, 1fr));
            gap: 20px;
            margin-bottom: 30px;
        }
        
        .stat-card {
            background-color: var(--card-bg);
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
            display: flex;
            flex-direction: column;
            align-items: center;
            text-align: center;
            transition: transform 0.3s ease;
        }
        
        .stat-card:hover {
            transform: translateY(-5px);
            box-shadow: 0 6px 12px rgba(0, 0, 0, 0.1);
        }
        
        .stat-icon {
            font-size: 2.5rem;
            margin-bottom: 15px;
            color: var(--primary-color);
        }
        
        .stat-value {
            font-size: 2rem;
            font-weight: 700;
            margin-bottom: 5px;
        }
        
        .stat-label {
            color: var(--light-text);
            font-size: 0.9rem;
        }
        
        .devices-container {
            background-color: var(--card-bg);
            border-radius: 10px;
            padding: 20px;
            box-shadow: 0 4px 6px rgba(0, 0, 0, 0.05);
            overflow-x: auto;
        }
        
        .devices-header {
            display: flex;
            flex-direction: column;
            gap: 10px;
            margin-bottom: 20px;
            padding-bottom: 10px;
            border-bottom: 1px solid var(--border-color);
        }
        
        @media (min-width: 768px) {
            .devices-header {
                flex-direction: row;
                justify-content: space-between;
                align-items: center;
            }
        }
        
        .devices-title {
            font-size: 1.3rem;
            font-weight: 600;
            display: flex;
            align-items: center;
            gap: 10px;
        }
        
        .devices-grid {
            display: grid;
            grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
            gap: 15px;
        }
        
        .device-card {
            background-color: var(--card-bg);
            border-radius: 8px;
            padding: 15px;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
            border: 1px solid var(--border-color);
        }
        
        .device-card:hover {
            transform: translateY(-3px);
            box-shadow: 0 6px 12px rgba(0, 0, 0, 0.1);
        }
        
        .device-info {
            display: flex;
            flex-direction: column;
            margin-bottom: 10px;
        }
        
        .device-info:last-child {
            margin-bottom: 0;
        }
        
        .device-label {
            font-size: 0.8rem;
            color: var(--light-text);
            margin-bottom: 5px;
            display: flex;
            align-items: center;
            gap: 5px;
        }
        
        .device-value {
            font-weight: 500;
            word-break: break-word;
        }
        
        .status-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 10px;
            border-radius: 20px;
            font-size: 0.8rem;
            font-weight: 600;
            width: fit-content;
        }
        
        .status-online {
            background-color: rgba(46, 204, 113, 0.1);
            color: var(--online-color);
        }
        
        .status-offline {
            background-color: rgba(231, 76, 60, 0.1);
            color: var(--offline-color);
        }
        
        .last-seen {
            font-size: 0.8rem;
            color: var(--light-text);
        }
        
        .no-data {
            text-align: center;
            padding: 40px;
            color: var(--light-text);
            grid-column: 1 / -1;
        }
        
        .error-message {
            text-align: center;
            padding: 20px;
            background-color: #ffebee;
            color: #c62828;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .success-message {
            text-align: center;
            padding: 20px;
            background-color: #e8f5e9;
            color: #2e7d32;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 10px;
        }
        
        .clear-btn {
            background-color: var(--danger-color);
            color: white;
            border: none;
            padding: 10px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 0.9rem;
            display: flex;
            align-items: center;
            gap: 8px;
            transition: all 0.3s ease;
            margin: 0 auto;
        }
        
        .clear-btn:hover {
            background-color: #c0392b;
            transform: translateY(-2px);
        }
        
        .clear-btn:active {
            transform: translateY(0);
        }
        
        .clear-form {
            text-align: center;
            margin: 20px 0;
        }
        
        .confirmation-dialog {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background-color: rgba(0, 0, 0, 0.5);
            z-index: 1000;
            justify-content: center;
            align-items: center;
        }
        
        .dialog-content {
            background-color: white;
            padding: 25px;
            border-radius: 10px;
            max-width: 500px;
            width: 90%;
            box-shadow: 0 5px 15px rgba(0, 0, 0, 0.3);
            text-align: center;
        }
        
        .dialog-actions {
            display: flex;
            justify-content: center;
            gap: 15px;
            margin-top: 20px;
        }
        
        .dialog-btn {
            padding: 8px 20px;
            border-radius: 5px;
            cursor: pointer;
            font-size: 0.9rem;
            border: none;
            transition: all 0.2s ease;
        }
        
        .dialog-btn-confirm {
            background-color: var(--danger-color);
            color: white;
        }
        
        .dialog-btn-cancel {
            background-color: var(--border-color);
            color: var(--text-color);
        }
        
        .dialog-btn:hover {
            opacity: 0.9;
            transform: translateY(-1px);
        }
        
        /* Mobile First Styles */
        @media (max-width: 767px) {
            .container {
                padding: var(--mobile-padding);
            }
            
            h1 {
                font-size: 1.5rem;
                flex-direction: column;
                gap: 10px;
            }
            
            .stats-container {
                grid-template-columns: 1fr;
            }
            
            .stat-card {
                padding: 15px;
            }
            
            .devices-container {
                padding: 15px;
            }
            
            .device-card {
                padding: 12px;
            }
        }
        
        /* Tablet Styles */
        @media (min-width: 768px) and (max-width: 1023px) {
            .stats-container {
                grid-template-columns: repeat(2, 1fr);
            }
            
            .devices-grid {
                grid-template-columns: repeat(2, 1fr);
            }
        }
        
        /* Desktop Styles */
        @media (min-width: 1024px) {
            .devices-grid {
                grid-template-columns: repeat(3, 1fr);
            }
        }
        
        /* Large Desktop */
        @media (min-width: 1440px) {
            .devices-grid {
                grid-template-columns: repeat(4, 1fr);
            }
        }
    </style>
</head>
<body>

<div class="container">
    <h1><i class="fa fa-plug"></i> Dashboard de Conexões</h1>
    
    <?php if (isset($error)): ?>
        <div class="error-message">
            <i class="fa fa-exclamation-circle"></i> <?php echo $error; ?>
        </div>
    <?php endif; ?>
    
    <?php if (isset($success_message)): ?>
        <div class="success-message">
            <i class="fa fa-check-circle"></i> <?php echo $success_message; ?>
        </div>
    <?php endif; ?>
    
    <div class="stats-container">
        <div class="stat-card">
            <div class="stat-icon">
                <i class="fa fa-desktop"></i>
            </div>
            <div class="stat-value">
                <?php echo count($connections); ?>
            </div>
            <div class="stat-label">Dispositivos Registrados</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">
                <i class="fa fa-check-circle"></i>
            </div>
            <div class="stat-value">
                <?php echo count(array_filter($connections, fn($c) => $c['is_online'])); ?>
            </div>
            <div class="stat-label">Dispositivos Online</div>
        </div>
        
        <div class="stat-card">
            <div class="stat-icon">
                <i class="fa fa-clock-o"></i>
            </div>
            <div class="stat-value">
                <?php echo count(array_filter($connections, fn($c) => !$c['is_online'])); ?>
            </div>
            <div class="stat-label">Dispositivos Offline</div>
        </div>
    </div>
    
    <div class="clear-form">
        <button type="button" class="clear-btn" id="clearDataBtn">
            <i class="fa fa-trash"></i> Limpar Todos os Dados
        </button>
    </div>
    
    <div class="devices-container">
        <div class="devices-header">
            <div class="devices-title">
                <i class="fa fa-list"></i> Lista de Dispositivos
            </div>
            <div class="last-update">
                Atualizado em: <?php echo date('d/m/Y H:i:s'); ?>
            </div>
        </div>
        
        <?php if (empty($connections)): ?>
            <div class="no-data">
                <i class="fa fa-database" style="font-size: 2rem; margin-bottom: 15px;"></i>
                <p>Nenhum dispositivo registrado no banco de dados.</p>
            </div>
        <?php else: ?>
            <div class="devices-grid">
                <?php foreach ($connections as $device): ?>
                    <div class="device-card">
                        <div class="device-info">
                            <div class="device-label">
                                <i class="fa fa-id-card"></i> ID do Dispositivo
                            </div>
                            <div class="device-value">
                                <?php echo htmlspecialchars($device['device_id']); ?>
                            </div>
                        </div>
                        
                        <div class="device-info">
                            <div class="device-label">
                                <i class="fa fa-mobile"></i> Modelo
                            </div>
                            <div class="device-value">
                                <?php echo htmlspecialchars($device['device_model']); ?>
                            </div>
                        </div>
                        
                        <div class="device-info">
                            <div class="device-label">
                                <i class="fa fa-signal"></i> Status
                            </div>
                            <div class="device-value">
                                <span class="status-badge <?php echo $device['is_online'] ? 'status-online' : 'status-offline'; ?>">
                                    <i class="fa <?php echo $device['is_online'] ? 'fa-check' : 'fa-times'; ?>"></i>
                                    <?php echo $device['is_online'] ? 'Online' : 'Offline'; ?>
                                </span>
                            </div>
                        </div>
                        
                        <div class="device-info">
                            <div class="device-label">
                                <i class="fa fa-clock-o"></i> Última Conexão
                            </div>
                            <div class="device-value">
                                <?php echo date('d/m/Y H:i:s', strtotime($device['last_connection'])); ?>
                            </div>
                        </div>
                    </div>
                <?php endforeach; ?>
            </div>
        <?php endif; ?>
    </div>
</div>

<!-- Confirmation Dialog -->
<div class="confirmation-dialog" id="confirmationDialog">
    <div class="dialog-content">
        <h3><i class="fa fa-exclamation-triangle" style="color: var(--danger-color);"></i> Confirmar Ação</h3>
        <p>Tem certeza que deseja limpar todos os dados de conexão? Esta ação não pode ser desfeita.</p>
        <div class="dialog-actions">
            <form method="post" style="display: inline;">
                <button type="submit" name="clear_data" class="dialog-btn dialog-btn-confirm">
                    <i class="fa fa-trash"></i> Limpar
                </button>
            </form>
            <button type="button" class="dialog-btn dialog-btn-cancel" id="cancelClearBtn">
                <i class="fa fa-times"></i> Cancelar
            </button>
        </div>
    </div>
</div>

<script>
    // Atualiza a página a cada 30 segundos
    setTimeout(function(){
        window.location.reload();
    }, 30000);
    
    // Mostrar diálogo de confirmação
    document.getElementById('clearDataBtn').addEventListener('click', function() {
        document.getElementById('confirmationDialog').style.display = 'flex';
    });
    
    // Cancelar limpeza
    document.getElementById('cancelClearBtn').addEventListener('click', function() {
        document.getElementById('confirmationDialog').style.display = 'none';
    });
    
    // Fechar diálogo ao clicar fora
    document.getElementById('confirmationDialog').addEventListener('click', function(e) {
        if (e.target === this) {
            this.style.display = 'none';
        }
    });
</script>

</body>
</html>

<?php include './includes/footer.php'; ?>